import re
from typing import List
from pipelines.prompta.rag import OBJECT_BY_CATEGORY, META_DATA


class Event:

    def __init__(self, identifier: str, attributes: List[str], symbol_format: str=''):
        if symbol_format == '':
            symbol_format = ' '.join((identifier, '{}'))
        self.symbol_format = symbol_format
        self.symbol_pattern = symbol_format.replace('{}', '([\d\S]+)')
        self.identifier = identifier
        self.attributes = attributes

    def to_symbol(self, *args) -> str:
        return self.symbol_format.format(*args)
    
    def accept(self, symbol: str) -> bool:
        match = re.match(self.symbol_pattern, symbol)
        if not match:
            return False
        return match.groups()[0] in self.attributes
    
    def copy(self, symbol: str):
        cls = self.__class__
        newone = cls.__new__(cls)
        newone.__init__(self.identifier, self.attributes, self.symbol_format)
        return newone
    
class InventoryEvent(Event):

    def __init__(self, identifier: str, attributes: List[str], symbol_format: str='', threshold: int=4):
        super().__init__(identifier, attributes, symbol_format)
        self.threshold = threshold

    def accept(self, symbol: str) -> bool:
        match = re.search(self.symbol_pattern, symbol)
        if not match:
            return False
        quantity, item = match.groups()
        return int(quantity) >= self.threshold and item in self.attributes
    
    def copy(self, symbol: str):
        cls = self.__class__
        newone = cls.__new__(cls)
        quantity, item = re.search(self.symbol_pattern, symbol).groups()
        newone.__init__(self.identifier, self.attributes, self.symbol_format, int(quantity))
        return newone

TimeOfDayEvent = Event("time_of_day", ["sunrise", "noon", "sunset", "midnight"])
BiomeEvent = Event("biome", META_DATA.biomesByName.keys())
HealthEvent = Event("health", ["good", "fair", "poor", "agonal"])
FoodEvent = Event("food", ["full", "low", "starving", "empty"])
EntityEvent = Event("entity", META_DATA.entitiesByName.keys(), "{} found")
OwnEvent = InventoryEvent("own", META_DATA.itemsByName.keys(), "own {} of {}", threshold=0)
GaindEvent = InventoryEvent("gained", META_DATA.itemsByName.keys(), "gained {} of {}", 0)
LostEvent = InventoryEvent("lost", META_DATA.itemsByName.keys(), "lost {} of {}", 0)
FindEvent = Event("exploreUntilFind", META_DATA.blocksByName.keys(), "exploreUntilFind{} succeeded")

EVENTS = [
    TimeOfDayEvent,
    BiomeEvent,
    HealthEvent,
    FoodEvent,
    EntityEvent,
    OwnEvent,
    GaindEvent,
    LostEvent,
    FindEvent
]

def is_event(symbol: str) -> bool:
    for event in EVENTS:
        if event.accept(symbol):
            return True
    return False